%{
/********************************************************************************
* Copyright (c) Erik Kunze 1997 - 1999
*
* Permission to use, distribute, and sell this software and its documentation
* for any purpose is hereby granted without fee, provided that the above
* copyright notice appear in all copies and that both that copyright notice and
* this permission notice appear in supporting documentation, and that the name
* of the copyright holder not be used in advertising or publicity pertaining to
* distribution of the software without specific, written prior permission.  The
* copyright holder makes no representations about the suitability of this
* software for any purpose.  It is provided "as is" without express or implied
* warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
*
* THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
* EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
* PERFORMANCE OF THIS SOFTWARE.
*
* Author: Erik Kunze
*******************************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#ifndef lint
static char monitor1_y[] = "$Id: monitor1.y,v 4.12 1999/01/14 16:53:03 erik Rel $";
#endif
#include <X11/keysym.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include "z80.h"
#include "resource.h"
#include "mem.h"
#include "io.h"
#include "machine.h"
#include "breakpoint.h"
#include "dialog.h"
#include "dis.h"
#include "screen.h"
#include "monitor.h"
#define E_OK			0
#define E_CMD_UNKNOWN	1
#define E_PARAM_INVAL	2
#define E_PARAM_RANGE 	3
#define E_BP_INUSE		4
#define E_BP_UNKNOWN	5
#define E_BP_NONE		6
#define E_HELP 			7
#ifdef DEBUG
#define YYDEBUG			1
#endif
struct reginfo {
uns8 *high_reg;
uns8 *low_reg;
int x;
int y;
};
static int yyparse (void);
static int yyerror(const char *);
static void dbgPrintString(const char *);
static void dbgPrintPanel(void);
static void dbgDisassemble(uns16);
static uns16 dbgGetMnemo(uns16);
int NumberBase = 10;
char *CmdLine;
static char buf[COLS + 1];
static int leaveDebugger;
static uns16 memoryPtr = 0;
static int activeReg = 0;
static struct reginfo regs[] = {
{ &A,   &F,    2,  2 },
{ &A1,  &F1,  13,  2 },
{ &B,   &C,    2,  3 },
{ &B1,  &C1,  13,  3 },
{ &D,   &E,    2,  4 },
{ &D1,  &E1,  13,  4 },
{ &H,   &L,    2,  5 },
{ &H1,  &L1,  13,  5 },
{ &HIX, &LIX,  2,  6 },
{ &HIY, &LIY,  2,  7 },
{ &HPC, &LPC,  2,  8 },
{ &HSP, &LSP,  2,  9 },
{ &I,   &R,   13,  7 }
};
%}
%token BASE
%token COPY
%token DIS
%token EQUAL
%token FILL
%token GO
%token INT_MODE
%token INT_FF1
%token INT_FF2
%token MINUS
%token NL
%token NUMBER
%token PLUS
%token QUIT
%token REGISTER
%token BP_ADD
%token BP_REMOVE
%token BP_SHOW
%token YACC_DEBUG
%token HELP
%token AY
%token UNKNOWN
%%
line
: NL
| command
;
command
: register
| memory
| fill
| copy
| dis
| go
| quit
| number_base
| interrupt
| breakpoint
| yacc_debug
| help
| ay
| error NL
{
return E_CMD_UNKNOWN;
}
;
register
: REGISTER NL
{
activeReg = ++activeReg % (sizeof(regs)/sizeof(regs[0]));
}
| REGISTER word NL
{
*(regs[activeReg].high_reg) = HIGHBYTE($2);
*(regs[activeReg].low_reg) = LOWBYTE($2);
}
;
memory
: PLUS NL
{
memoryPtr++;
}
| PLUS word NL
{
memoryPtr += $2;
}
| MINUS NL
{
memoryPtr--;
}
| MINUS word NL
{
memoryPtr -= $2;
}
| EQUAL NL
{
memoryPtr =
(*(regs[activeReg].high_reg) << 8) | *(regs[activeReg].low_reg);
}
| EQUAL word NL
{
memoryPtr = (uns16)$2;
}
| word NL
{
if ($1 <= 0xff)
{
MEM(memoryPtr) = (uns8)$1;
memoryPtr++;
}
else
{
MEM(memoryPtr) = (uns8)($1 & 0xff);
memoryPtr++;
MEM(memoryPtr) = (uns8)(($1 >> 8) & 0xff);
memoryPtr++;
}
}
;
fill
: FILL word word NL
{
uns16 i;
for (i = (uns16)$2; i <= (uns16)$3; i++)
{
MEM(i) = 0;
}
}
| FILL word word byte NL
{
uns16 i;
if ($4 > 0xff)
{
return E_PARAM_INVAL;
}
else
{
for (i = (uns16)$2; i <= (uns16)$3; i++)
{
MEM(i) = (uns8)$4;
}
}
}
;
copy
: COPY word word word NL
{
uns16 src = (uns16)$2, dst = (uns16)$3, cnt = (uns16)$4;
if (src + cnt < dst || src > dst)
{
for (; cnt > 0; cnt--, dst++, src++)
{
MEM(dst) = MEM(src);
}
}
else
{
src += cnt;
dst += cnt;
for (; cnt > 0; cnt--, dst--, src--)
{
MEM(dst) = MEM(src);
}
}
}
;
dis
: DIS NL
{
dbgDisassemble(PC);
}
| DIS word NL
{
dbgDisassemble((uns16)$2);
}
;
go
: GO NL
{
leaveDebugger = 1;
}
| GO word NL
{
PC = (uns16)$2;
leaveDebugger = 1;
}
;
quit
: QUIT NL
{
leaveDebugger = 1;
}
;
number_base
: BASE NL
{
(void)sprintf(buf, "%d", NumberBase);
dbgPrintString(buf);
}
| BASE number NL
{
if ($2 == 2 || $2 == 8 || $2 == 10 || $2 == 16)
{
NumberBase = $2;
}
else
{
return E_PARAM_RANGE;
}
}
;
interrupt
: INT_MODE byte NL
{
if ($2 >= 0 && $2 <= 2)
{
IM = (uns8)$2;
}
else
{
return E_PARAM_RANGE;
}
}
| INT_FF1 byte NL
{
if ($2 >= 0 && $2 <= 1)
{
IFF1 = (uns8)$2;
}
else
{
return E_PARAM_RANGE;
}
}
| INT_FF2 byte NL
{
if ($2 >= 0 && $2 <= 1)
{
IFF2 = (uns8)$2;
}
else
{
return E_PARAM_RANGE;
}
}
;
breakpoint
: BP_ADD byte word NL
{
int maxpage = GETCFG(machine) < SP_128 ? 2 : 7;
#ifdef XZX_SCORPION
maxpage += GETCFG(machine) == SP_SCORPION ? 8 : 0;
#endif
if ($2 < 0 || $2 > maxpage || $3 < 0 || $3 > 0x3fff)
{
return E_PARAM_RANGE;
}
if (maxpage == 2)
{
if ($2 != 0 && $2 != 2 && $2 != 5)
{
return E_PARAM_RANGE;
}
}
if (BpSet(RAM0 + $2, (uns16)$3) == -1U)
{
return E_BP_INUSE;
}
}
| BP_REMOVE number NL
{
if (BpRetrieve((unsigned int)$2))
{
BpUnset((unsigned int)$2);
}
else
{
return E_BP_UNKNOWN;
}
}
| BP_REMOVE NL
{
if (!BpUnsetAll())
{
return E_BP_NONE;
}
}
| BP_SHOW number NL
{
Breakpoint *bpp = BpRetrieve((unsigned int)$2);
if (bpp)
{
(void)sprintf(buf, "%X:%04X  %s", bpp->page - RAM0, bpp->addr,
bpp->opstr);
dbgPrintString(buf);
}
else
{
return E_BP_UNKNOWN;
}
}
| BP_SHOW NL
{
Breakpoint *bpp;
unsigned int bpno = 0;
int row = 0;
if (!BpRetrieve(0))
{
return E_BP_NONE;
}
ClearScreen();
while ((bpp = BpRetrieve(bpno)))
{
(void)sprintf(buf, "%04d  %X:%04X  %s", bpno, bpp->page - RAM0,
bpp->addr, bpp->opstr);
SetCursor(0, row);
PrintString(buf);
bpno++;
row++;
if (row == ROWS)
{
row = 0;
ScreenRefresh();
if (GetKey() == XK_Escape)
{
return E_OK;
}
ClearScreen();
}
}
if (row)
{
ScreenRefresh();
(void)GetKey();
}
}
;
yacc_debug
: YACC_DEBUG number NL
{
#ifdef DEBUG
yydebug = (int)$2;
#else
return E_CMD_UNKNOWN;
#endif
}
;
help
: HELP NL
{
return E_HELP;
}
;
ay
: AY NL
{
int row, i;
ClearScreen();
for (i = 0, row = 0; i <= 15; i += 4, row++)
{
(void)sprintf(buf, "%02d %02X  %02d %02X  %02d %02X  %02d %02X",
i, Psg[i], i + 1, Psg[i + 1], i + 2, Psg[i + 2],
i + 3, Psg[i + 3]);
SetCursor(0, row);
PrintString(buf);
}
ScreenRefresh();
(void)GetKey();
}
| AY number NL
{
if ((unsigned int)$2 <= 15)
{
(void)sprintf(buf, "%02X", Psg[$2]);
dbgPrintString(buf);
}
else
{
return E_PARAM_RANGE;
}
}
| AY number byte NL
{
if ((unsigned int)$2 <= 15)
{
Machine->outPort(P_SNDCONTROL, (uns8)$2);
Machine->outPort(P_SNDDATA, (uns8)$3);
}
else
{
return E_PARAM_RANGE;
}
}
;
number
: NUMBER
{
$$ = $1;
}
| error NL
{
return E_PARAM_INVAL;
}
;
byte
: number
{
if ((unsigned int)$1 <= 0x00ffU)
{
$$ = $1;
}
else
{
return E_PARAM_RANGE;
}
}
;
word
: number
{
if ((unsigned int)$1 <= 0xffffU)
{
$$ = $1;
}
else
{
return E_PARAM_RANGE;
}
}
;
%%
static int
yyerror(const char *s)
{
/* YYABORT; disabled by Yarek - causes compilation crash */
}
void
Debugger(int why)
{
int oldInDialog;
char input[COLS];
const char *errmsg;
int rc;
if (why == DBG_BREAKPOINT)
{
uns16 pc = PC - 1;
unsigned int bpno;
if ((bpno = BpLookup(RPAGE(pc), pc)) == -1U)
{
TSTATES += 11;
PUSH(HPC, LPC);
PC = 0x0000;
return;
}
}
oldInDialog = EnterOSD();
SetAttr((BLACK << 3) | WHITE);
SetBorderColor(BLACK);
for (leaveDebugger = 0; leaveDebugger == 0;)
{
dbgPrintPanel();
do
{
SetCursor(0, 23);
PrintLetter('>');
}
while (!GetString(input, sizeof(input) - 1, COLS - 1));
(void)strcat(input, "\n");
CmdLine = input;
yylex_init();
if ((rc = yyparse()) != E_OK)
{
switch (rc)
{
default:
case E_CMD_UNKNOWN:
errmsg = "unknown command";
break;
case E_PARAM_INVAL:
errmsg = "invalid argument";
break;
case E_PARAM_RANGE:
errmsg = "invalid value";
break;
case E_BP_INUSE:
errmsg = "breakpoint already set";
break;
case E_BP_UNKNOWN:
errmsg = "unknown breakpoint";
break;
case E_BP_NONE:
errmsg = "no breakpoints set";
break;
case E_HELP:
errmsg = "help? read the manual!";
break;
}
dbgPrintString(errmsg);
}
}
SetBorderColor((MENU_TEXT_ATTR & PAPER) >> 3);
LeaveOSD(oldInDialog);
}
static void
dbgPrintString(const char *string)
{
SetCursor(1, 23);
DrawBox(COLS - 2, 1);
PrintString(string);
ScreenRefresh();
(void)GetKey();
}
static void
dbgPrintPanel(void)
{
int i;
uns16 mem;
ClearScreen();
(void)dbgGetMnemo(PC);
PrintString(buf);
SetCursor(0, 2);
(void)sprintf(buf, "AF %04X   AF' %04X      SZXHXVPC", AF, AF1);
PrintString(buf);
SetCursor(0, 3);
(void)sprintf(buf, "BC %04X   BC' %04X   F  ", BC, BC1);
PrintString(buf);
for (i = 0x80; i > 0; i >>= 1)
{
PrintLetter(F & i ? '1' : '0');
}
SetCursor(0, 4);
(void)sprintf(buf, "DE %04X   DE' %04X   F' ", DE, DE1);
PrintString(buf);
for (i = 0x80; i > 0; i >>= 1)
{
PrintLetter(F1 & i ? '1' : '0');
}
SetCursor(0, 5);
(void)sprintf(buf, "HL %04X   HL' %04X", HL, HL1);
PrintString(buf);
SetCursor(0, 6);
(void)sprintf(buf, "IX %04X              IM 1 2 RMVL", IX);
PrintString(buf);
SetCursor(0, 7);
(void)sprintf(buf, "IY %04X   IR  %02X%02X    %1d %1d %1d ",
IX,I, R, IM, IFF1, IFF2);
PrintString(buf);
switch (GETCFG(machine))
{
case SP_48_2:
case SP_48_3:
(void)sprintf(buf, "%1d---", RPAGE(0));
break;
case SP_128:
(void)sprintf(buf, "%1d%1d%1d%1d",
RPAGE(0) - SP128ROM0,
Last0x7FFD & B_SELRAM,
Last0x7FFD & B_SELSCREEN ? 7 : 5,
(Last0x7FFD & B_PAGING) >> 5);
break;
#ifdef XZX_PLUS3
case SP_3:
if (!EXTENDED_RAM(Last0x1FFD))
{
(void)sprintf(buf, "%1d%1d%1d%1d",
RPAGE(0) - SP3ROM0,
Last0x7FFD & B_SELRAM,
Last0x7FFD & B_SELSCREEN ? 7 : 5,
(Last0x7FFD & B_PAGING) >> 5);
}
else
{
(void)sprintf(buf, "-%1d-0", (Last0x1FFD & B_SELRAM3) >> 1);
}
break;
#endif
#ifdef XZX_PENTAGON
case SP_PENTAGON:
(void)sprintf(buf, "%1d%1d%1d%1d",
RPAGE(0) - PENTROM0,
Last0x7FFD & B_SELRAM,
Last0x7FFD & B_SELSCREEN ? 7 : 5,
(Last0x7FFD & B_PAGING) >> 5);
break;
#endif
#ifdef XZX_SCORPION
case SP_SCORPION:
if (!(Last0x1FFD & 0x1))
{
(void)sprintf(buf, "%1d%1X%1d%1d",
RPAGE(0) - SCORROM0,
(Last0x7FFD & B_SELRAM) |
((Last0x1FFD & 0x10) >> 1),
Last0x7FFD & B_SELSCREEN ? 7 : 5,
(Last0x7FFD & B_PAGING) >> 5);
}
else
{
(void)sprintf(buf, "-%1X%1d%1d",
(Last0x7FFD & B_SELRAM) |
((Last0x1FFD & 0x10) >> 1),
Last0x7FFD & B_SELSCREEN ? 7 : 5,
(Last0x7FFD & B_PAGING) >> 5);
}
break;
#endif
}
PrintString(buf);
SetCursor(0, 8);
(void)sprintf(buf, "PC %04X", PC);
PrintString(buf);
SetCursor(0, 9);
(void)sprintf(buf, "SP %04X", SP);
PrintString(buf);
for (mem = memoryPtr - 12, i = 0; i < 8; mem++, i++)
{
SetCursor(0, 13 + i);
(void)sprintf(buf, "%04X %02X   %04X %02X   %04X %02X",
mem, MEM(mem),
(uns16)(mem + 8), MEM((uns16)(mem + 8)),
(uns16)(mem + 16), MEM((uns16)(mem +16)));
PrintString(buf);
}
SetCursor(9, 17);
PrintLetter('>');
SetCursor(17, 17);
PrintLetter('<');
SetCursor(regs[activeReg].x, regs[activeReg].y);
PrintLetter('*');
ScreenRefresh();
}
static void
dbgDisassemble(uns16 addr)
{
int i;
do
{
ClearScreen();
for (i = 0; i < ROWS; i++)
{
SetCursor(0, i);
addr = dbgGetMnemo(addr);
PrintString(buf);
}
ScreenRefresh();
}
while (GetKey() != XK_Escape);
}
static uns16
dbgGetMnemo(uns16 addr)
{
char *p;
uns16 i, j;
p = buf;
(void)sprintf(p, "%04X  ", addr);
p += 6;
j = Disassemble(buf + 16, addr) - addr;
for (i = 0; i < j; i++)
{
(void)sprintf(p, "%02X", MEM((uns16)(addr + i)));
p += 2;
}
while (p < &buf[16])
{
*p++ = ' ';
}
return (addr + j);
}
